const db = require('../config/database');
const Transaction = require('../models/Transaction');

class WebhookController {
  static async handleBankNotification(req, res) {
    try {
      const { bank_code } = req.params;
      const payload = req.body;
      const headers = req.headers;
      const ip = req.ip;

      // Validar API KEY del banco
      const apiKey = headers['api-key'];
      if (!await validateBankApiKey(bank_code, apiKey)) {
        return res.status(403).json({
          codigo: "99",
          mensajeCliente: "corrija el API KEY",
          mensajeSistema: "Error en API KEY"
        });
      }

      // Registrar webhook
      const [result] = await db.execute(
        `INSERT INTO webhook_logs (bank_id, payload, headers, ip_address) 
         VALUES ((SELECT id FROM banks WHERE code = ?), ?, ?, ?)`,
        [bank_code, JSON.stringify(payload), JSON.stringify(headers), ip]
      );

      // Procesar la notificación
      await processBankNotification(bank_code, payload);

      // Verificar si es una renotificación
      const isRenotification = await checkReNotification(payload);

      if (isRenotification) {
        return res.json({
          codigo: "01",
          mensajeCliente: "pago previamente recibido",
          mensajeSistema: "Renotificado"
        });
      }

      res.json({
        codigo: "00",
        mensajeCliente: "Aprobado",
        mensajeSistema: "Notificado"
      });

    } catch (error) {
      console.error('Error en webhook:', error);
      res.status(500).json({
        codigo: "98",
        mensajeCliente: "Error interno",
        mensajeSistema: "Error procesando notificación"
      });
    }
  }
}

async function validateBankApiKey(bankCode, apiKey) {
  // Implementar validación específica por banco
  const bank = await Bank.findByCode(bankCode);
  const config = await Bank.getConfig(bank.id);
  
  return config.webhook_api_key === apiKey;
}

async function processBankNotification(bankCode, payload) {
  // Procesar según el banco
  switch (bankCode) {
    case '0102': // BDV
      await processBDVNotification(payload);
      break;
    // Otros bancos...
  }
}

async function processBDVNotification(payload) {
  // Buscar transacción por referencia o crear una nueva
  const reference = payload.referenciaBancoOrdenante;
  
  if (reference) {
    await Transaction.updateStatus(reference, 'completed', {
      code: '00',
      message: 'Pago confirmado via webhook'
    });
  }
}

async function checkReNotification(payload) {
  // Verificar si ya hemos procesado esta notificación
  const [logs] = await db.execute(
    `SELECT COUNT(*) as count FROM webhook_logs 
     WHERE payload->>'$.referenciaBancoOrdenante' = ? 
     AND processed = true`,
    [payload.referenciaBancoOrdenante]
  );

  return logs[0].count > 0;
}

module.exports = WebhookController;